'use client'

import Image from 'next/image'
import Link from 'next/link'
import { useState } from 'react'
import { Product } from '@/lib/data/products'

interface ProductCardProps {
  product: Product;
}

export default function ProductCard({ product }: ProductCardProps) {
  const [imageError, setImageError] = useState(false)
  const placeholderUrl = `https://via.placeholder.com/400x400/f3f4f6/9ca3af?text=${encodeURIComponent(product.name)}`

  return (
    <div className="group relative">
      <div className="card hover:scale-105 transition-transform">
        {/* Badge */}
        {product.badge && (
          <div className="absolute top-2 right-2 z-10">
            <span className="bg-red-600 text-white px-2 py-1 rounded-full text-xs font-semibold">
              {product.badge}
            </span>
          </div>
        )}
        
        {/* Image Container */}
        <div className="relative aspect-square bg-gray-100 rounded-lg mb-4 overflow-hidden">
          <Image
            src={imageError ? placeholderUrl : product.image}
            alt={product.name}
            fill
            className="object-cover group-hover:scale-110 transition-transform duration-300"
            sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
            onError={() => setImageError(true)}
          />
          
          {/* Stock Status */}
          {!product.inStock && (
            <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
              <span className="bg-white/90 text-gray-900 px-4 py-2 rounded-lg font-semibold">
                Out of Stock
              </span>
            </div>
          )}
        </div>
        
        {/* Product Info */}
        <div className="space-y-2">
          <p className="text-xs text-gray-500 uppercase tracking-wide">
            {product.category}
          </p>
          <h3 className="font-semibold text-lg text-gray-900 line-clamp-2">
            {product.name}
          </h3>
          <p className="text-sm text-gray-600 line-clamp-2">
            {product.description}
          </p>
          <p className="text-red-600 font-bold text-xl">
            {product.price}
          </p>
        </div>
        
        {/* Actions */}
        <div className="mt-4 space-y-2">
          <button 
            className={`w-full py-2 rounded-lg font-semibold transition-colors ${
              product.inStock 
                ? 'bg-gray-900 text-white hover:bg-gray-800' 
                : 'bg-gray-200 text-gray-400 cursor-not-allowed'
            }`}
            disabled={!product.inStock}
          >
            {product.inStock ? 'Add to Cart' : 'Out of Stock'}
          </button>
          <Link 
            href={`/shop/product/${product.id}`}
            className="block text-center text-sm text-gray-600 hover:text-red-600 transition-colors"
          >
            View Details →
          </Link>
        </div>
      </div>
    </div>
  )
}
